<?php
require_once '../config.php';
requireAdminAuth();

$action = $_GET['action'] ?? 'list';
$id = $_GET['id'] ?? 0;

// Handle delete action
if ($action === 'delete' && $id) {
    $stmt = $pdo->prepare("DELETE FROM articles WHERE id = ?");
    $stmt->execute([$id]);
    header('Location: ?page=articles&msg=deleted');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = $_POST['title'];
    $slug = $_POST['slug'];
    $content = $_POST['content'];
    $excerpt = $_POST['excerpt'];
    $category_id = $_POST['category_id'];
    $status = $_POST['status'];
    $author = $_POST['author'] ?? 'Admin';
    
    if ($id) {
        // Update existing article
        $stmt = $pdo->prepare("
            UPDATE articles 
            SET title = ?, slug = ?, content = ?, excerpt = ?, category_id = ?, status = ?, author = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$title, $slug, $content, $excerpt, $category_id, $status, $author, $id]);
        $message = 'updated';
    } else {
        // Insert new article
        $stmt = $pdo->prepare("
            INSERT INTO articles (title, slug, content, excerpt, category_id, status, author, published_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$title, $slug, $content, $excerpt, $category_id, $status, $author]);
        $id = $pdo->lastInsertId();
        $message = 'created';
    }
    
    header("Location: ?page=articles&action=edit&id=$id&msg=$message");
    exit;
}

// Get article data for edit
$article = null;
if ($action === 'edit' && $id) {
    $stmt = $pdo->prepare("SELECT * FROM articles WHERE id = ?");
    $stmt->execute([$id]);
    $article = $stmt->fetch();
}

// Get categories
$categories = $pdo->query("SELECT * FROM categories ORDER BY name")->fetchAll();

// Get articles list
$articles = $pdo->query("
    SELECT a.*, c.name as category_name 
    FROM articles a 
    LEFT JOIN categories c ON a.category_id = c.id 
    ORDER BY a.created_at DESC
")->fetchAll();
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">Kelola Artikel</h1>
    <a href="?page=articles&action=create" class="btn btn-primary">
        <i class="fas fa-plus"></i> Tambah Artikel
    </a>
</div>

<?php if (isset($_GET['msg'])): ?>
    <div class="alert alert-<?php echo $_GET['msg'] === 'deleted' ? 'warning' : 'success'; ?> alert-dismissible fade show">
        <?php
        switch ($_GET['msg']) {
            case 'created': echo 'Artikel berhasil dibuat!'; break;
            case 'updated': echo 'Artikel berhasil diperbarui!'; break;
            case 'deleted': echo 'Artikel berhasil dihapus!'; break;
        }
        ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($action === 'create' || $action === 'edit'): ?>
<!-- Form Create/Edit Article -->
<div class="admin-card">
    <h4><?php echo $action === 'create' ? 'Tambah Artikel Baru' : 'Edit Artikel'; ?></h4>
    
    <form method="POST">
        <div class="row">
            <div class="col-md-8">
                <div class="mb-3">
                    <label for="title" class="form-label">Judul Artikel</label>
                    <input type="text" class="form-control" id="title" name="title" 
                           value="<?php echo htmlspecialchars($article['title'] ?? ''); ?>" required>
                </div>
                
                <div class="mb-3">
                    <label for="slug" class="form-label">Slug URL</label>
                    <input type="text" class="form-control" id="slug" name="slug" 
                           value="<?php echo htmlspecialchars($article['slug'] ?? ''); ?>" required>
                    <div class="form-text">URL-friendly version of the title</div>
                </div>
                
                <div class="mb-3">
                    <label for="excerpt" class="form-label">Ringkasan</label>
                    <textarea class="form-control" id="excerpt" name="excerpt" rows="3"><?php echo htmlspecialchars($article['excerpt'] ?? ''); ?></textarea>
                </div>
                
                <div class="mb-3">
                    <label for="content" class="form-label">Konten</label>
                    <textarea class="form-control" id="content" name="content" rows="12"><?php echo htmlspecialchars($article['content'] ?? ''); ?></textarea>
                </div>
            </div>
            
            <div class="col-md-4">
                <div class="mb-3">
                    <label for="category_id" class="form-label">Kategori</label>
                    <select class="form-select" id="category_id" name="category_id" required>
                        <option value="">Pilih Kategori</option>
                        <?php foreach ($categories as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>" 
                                <?php echo ($article['category_id'] ?? '') == $cat['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($cat['name']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="mb-3">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-select" id="status" name="status" required>
                        <option value="draft" <?php echo ($article['status'] ?? '') === 'draft' ? 'selected' : ''; ?>>Draft</option>
                        <option value="published" <?php echo ($article['status'] ?? '') === 'published' ? 'selected' : ''; ?>>Published</option>
                    </select>
                </div>
                
                <div class="mb-3">
                    <label for="author" class="form-label">Penulis</label>
                    <input type="text" class="form-control" id="author" name="author" 
                           value="<?php echo htmlspecialchars($article['author'] ?? 'Admin'); ?>">
                </div>
                
                <?php if ($article): ?>
                <div class="card">
                    <div class="card-body">
                        <h6>Informasi Artikel</h6>
                        <p class="mb-1"><small>Dibuat: <?php echo date('d M Y H:i', strtotime($article['created_at'])); ?></small></p>
                        <p class="mb-1"><small>Diupdate: <?php echo date('d M Y H:i', strtotime($article['updated_at'])); ?></small></p>
                        <p class="mb-0"><small>Views: <?php echo $article['views']; ?></small></p>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="d-flex gap-2">
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save"></i> Simpan Artikel
            </button>
            <a href="?page=articles" class="btn btn-secondary">Batal</a>
        </div>
    </form>
</div>

<?php else: ?>
<!-- Articles List -->
<div class="admin-card">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h4>Daftar Artikel</h4>
        <span class="badge bg-primary">Total: <?php echo count($articles); ?> artikel</span>
    </div>
    
    <?php if (empty($articles)): ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle"></i> Belum ada artikel. 
            <a href="?page=articles&action=create" class="alert-link">Tambah artikel pertama</a>
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Judul</th>
                        <th>Kategori</th>
                        <th>Status</th>
                        <th>Tanggal</th>
                        <th>Views</th>
                        <th>Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($articles as $art): ?>
                    <tr>
                        <td>
                            <strong><?php echo htmlspecialchars($art['title']); ?></strong>
                            <br><small class="text-muted"><?php echo htmlspecialchars($art['slug']); ?></small>
                        </td>
                        <td><?php echo htmlspecialchars($art['category_name']); ?></td>
                        <td>
                            <span class="badge bg-<?php echo $art['status'] === 'published' ? 'success' : 'secondary'; ?>">
                                <?php echo ucfirst($art['status']); ?>
                            </span>
                        </td>
                        <td>
                            <small><?php echo date('d M Y', strtotime($art['created_at'])); ?></small>
                        </td>
                        <td><?php echo $art['views']; ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="?page=articles&action=edit&id=<?php echo $art['id']; ?>" 
                                   class="btn btn-outline-primary">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <a href="?page=articles&action=delete&id=<?php echo $art['id']; ?>" 
                                   class="btn btn-outline-danger btn-delete"
                                   onclick="return confirm('Yakin hapus artikel ini?')">
                                    <i class="fas fa-trash"></i>
                                </a>
                                <a href="../index.php?page=article&id=<?php echo $art['id']; ?>" 
                                   target="_blank" class="btn btn-outline-info">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>
<?php endif; ?>

<script>
// Auto-generate slug from title
document.addEventListener('DOMContentLoaded', function() {
    const titleInput = document.getElementById('title');
    const slugInput = document.getElementById('slug');
    
    if (titleInput && slugInput) {
        titleInput.addEventListener('input', function() {
            if (!slugInput.dataset.manual) {
                const slug = this.value
                    .toLowerCase()
                    .replace(/[^a-z0-9 -]/g, '')
                    .replace(/\s+/g, '-')
                    .replace(/-+/g, '-')
                    .trim();
                slugInput.value = slug;
            }
        });
        
        slugInput.addEventListener('input', function() {
            this.dataset.manual = 'true';
        });
    }
});
</script>